# 
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#     http://www.apache.org/licenses/LICENSE-2.0
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import os
from pprint import pformat

from monai import transforms
from monai.apps import get_logger
from monai.auto3dseg.utils import datafold_read
from monai.bundle import ConfigParser
from monai.utils import ensure_tuple, look_up_option

# Define the root path to the orignal manual label files, supervoxels, and pseudolabels.
all_base_dirs = {
    "AbdomenCT-1K": "/data/AbdomenCT-1K",
    "FLARE22": "/data/AbdomenCT-1K/FLARE22Train",
    "AMOS22": "/data/AMOS22",
    "BTCV-Abdomen": "/data/BTCV/Abdomen",
    "BTCV-Cervix": "/data/BTCV/Cervix",
    "CT-ORG": "/data/CT-ORG",
    "Multi-organ-Abdominal-CT-btcv": "/data/Multi-organ-Abdominal-CT/res_1.0mm_relabeled2",
    "Multi-organ-Abdominal-CT-tcia": "/data/Multi-organ-Abdominal-CT/res_1.0mm_relabeled2",
    "Pancreas-CT": "/data/Pancreas-CT",
    "Task06": "/data/Task06",
    "Task07": "/data/Task07",
    "Task08": "/data/Task08",
    "Task09": "/data/Task09",
    "Task10": "/data/Task10",
    "TotalSegmentator": "/data/TotalSegmentator",
    "TotalSegmentatorV2": "/data/TotalSegmentatorV2",
    "Task03": "/data/Task03",
    "Bone-NIH": "/data/Bone-NIH",
    "CRLM-CT": "/data/CRLM-CT/nifti",
    "VerSe": "/data/VerSe/",
    "AeroPath": "/data/AeroPath/",
    "CTPelvic1K-CLINIC": "/data/CTPelvic1K-CLINIC",
    "NLST": "/data/NLST",
    "LIDC": "/data/LIDC",
    "Covid19": "/data/Covid19",
    "TCIA_Colon": "/data/TCIA_Colon",
    "StonyBrook-CT": "/data/StonyBrook-CT",
}
# Notice the root path to supervoxel and pseudolabel has the same sub-folder structure as all_base_dirs
# The path is generated by replacing json_base with supervoxel_base/pl_base in get_datalist_with_dataset_name
json_base = "/data/"
supervoxel_base = "/workspace_infer/supervoxel_sam/"
pl_base = "/workspace_infer/V2_pseudo_12Feb2024/"


cur_json_dir = os.path.join(os.path.dirname(__file__), "jsons")
logger = get_logger(__name__)


def get_json_files_k_folds(json_dir=None, base_dirs=None, k=5):
    """the json files are generated by data/make_datalists.py, stored at `json_dir`"""
    if json_dir is None:
        json_dir = cur_json_dir
    if base_dirs is None:
        base_dirs = all_base_dirs
    output_dict = {
        item: os.path.join(json_dir, f"{item}_{k}_folds.json") for item in base_dirs
    }
    logger.debug(pformat(output_dict))
    return output_dict


def get_class_names(json_dir=None):
    """
    the list of class names, background is at 0
    """
    parser = ConfigParser.load_config_file(os.path.join(json_dir, "label_dict.json"))
    label_dict = dict(parser)
    label_dict["unspecified region"] = 0
    inv_label_dict = {v: k for k, v in label_dict.items()}
    label_list = []
    for i in range(len(label_dict)):
        label_list.append(inv_label_dict[i])
    return label_list


def get_datalist_with_dataset_name(
    datasets=None, fold_idx=-1, key="training", json_dir=None, base_dirs=None
):
    """
    when `datasets` is None, it returns a list of all data from all datasets.
    when `datasets` is a list of dataset names, it returns a list of all data from the specified datasets.

    train_list's item format::

        {"image": image_file_path, "label": label_file_path, "dataset_name": dataset_name, "fold": fold_id}

    """
    if base_dirs is None:
        base_dirs = all_base_dirs  # all_base_dirs is the broader set
    # get the list of training/validation files (absolute path)
    json_files = get_json_files_k_folds(json_dir=json_dir, base_dirs=base_dirs)
    if datasets is None:
        loading_dict = json_files.copy()
    else:
        loading_dict = {
            k: look_up_option(k, json_files) for k in ensure_tuple(datasets)
        }
    train_list, val_list = [], []
    for k, j in loading_dict.items():
        t, v = datafold_read(j, basedir=all_base_dirs[k], fold=fold_idx, key=key)
        for item in t:
            item["dataset_name"] = k
            if "label_sv" in item.keys():
                item["label_sv"] = item["label_sv"].replace(
                    json_base, supervoxel_base, 1
                )
            if "pseudo_label" in item.keys():
                item["pseudo_label"] = item["pseudo_label"].replace(
                    json_base, pl_base, 1
                )
        train_list += t
        for item in v:
            item["dataset_name"] = k
            if "label_sv" in item.keys():
                item["label_sv"] = item["label_sv"].replace(
                    json_base, supervoxel_base, 1
                )
            if "pseudo_label" in item.keys():
                item["pseudo_label"] = item["pseudo_label"].replace(
                    json_base, pl_base, 1
                )

        val_list += v
    logger.warning(
        f"data list from datasets={datasets} fold={fold_idx}: train={len(train_list)}, val={len(val_list)}"
    )
    return ensure_tuple(train_list), ensure_tuple(val_list)


def get_datalist_with_dataset_name_and_transform(
    image_key,
    label_key,
    label_sv_key,
    pseudo_label_key,
    num_patches_per_image,
    patch_size,
    datasets=None,
    fold_idx=-1,
    key="training",
    json_dir=None,
    base_dirs=None,
):
    """
    when `datasets` is None, it returns a list of all data from all datasets.
    when `datasets` is a list of dataset names, it returns a list of all data from the specified datasets.
    Return file lists and specific transforms for each dataset.

    """
    if base_dirs is None:
        base_dirs = all_base_dirs  # all_base_dirs is the broader set
    train_list, val_list = get_datalist_with_dataset_name(
        datasets=datasets,
        fold_idx=fold_idx,
        key=key,
        json_dir=json_dir,
        base_dirs=base_dirs,
    )
    # get the list of training/validation files (absolute path)
    json_files = get_json_files_k_folds(json_dir=json_dir, base_dirs=base_dirs)
    if datasets is None:
        loading_dict = json_files.copy()
    else:
        loading_dict = {
            k: look_up_option(k, json_files) for k in ensure_tuple(datasets)
        }

    dataset_transforms = {}
    dataset_transforms_val = {}
    for k, j in loading_dict.items():
        parser = ConfigParser()
        parser.read_config(j)
        # those parameters are required to initiate the transforms
        parser.update(
            pairs={
                "image_key": image_key,
                "label_key": label_key,
                "label_sv_key": label_sv_key,
                "pseudo_label_key": pseudo_label_key,
                "num_patches_per_image": num_patches_per_image,
                "patch_size": patch_size,
            }
        )
        transform = parser.get_parsed_content("training_transform")
        dataset_transforms[k] = transforms.Compose(transform)
        transform_val = parser.get_parsed_content("validation_transform", default=None)
        dataset_transforms_val[k] = (
            transforms.Compose(transform_val) if transform_val is not None else None
        )
    return (
        ensure_tuple(train_list),
        ensure_tuple(val_list),
        dataset_transforms,
        dataset_transforms_val,
    )


def compute_dataset_weights(datalist, weight_path="./data/dataset_weights.yaml"):
    """based on class-wise weight, assign a weight to each training sample"""
    cfg = ConfigParser.load_config_file(weight_path)
    w = []
    for item in datalist:
        fg_w = cfg[item["dataset_name"]]
        w.append(fg_w)
        item["w"] = fg_w
    return w


def calculate_dataset_weights(datalist):
    dataset_name = []
    dataset_counts = {}
    for item in datalist:
        dn = item["dataset_name"]
        if dn in dataset_name:
            dataset_counts[dn] += 1
        else:
            dataset_name.append(dn)
            dataset_counts[dn] = 1
    dataset_weights = {}
    non_tumor_count = 0
    tumor_count = 0
    for item in dataset_name:
        if item not in ["Task03", "Task06", "Task07", "Task08", "Task10", "Bone-NIH"]:
            non_tumor_count += dataset_counts[item]
        else:
            tumor_count += dataset_counts[item]

    for item in dataset_name:
        if item not in ["Task03", "Task06", "Task07", "Task08", "Task10", "Bone-NIH"]:
            dataset_weights[item] = 100 / dataset_counts[item]  # non_tumor_count
        else:
            dataset_weights[item] = 100 / dataset_counts[item]  # tumor_count

    dataset_prob = {}
    total_prob = 0
    for item in dataset_name:
        dataset_prob[item] = dataset_weights[item] * dataset_counts[item]
        total_prob += dataset_prob[item]
    for item in dataset_name:
        dataset_prob[item] /= total_prob

    import json

    with open("./dataset_counts.yaml", "w") as f:
        json.dump(dataset_counts, f, indent=4)
    with open("./dataset_weights.yaml", "w") as f:
        json.dump(dataset_weights, f, indent=4)
    with open("./dataset_prob.yaml", "w") as f:
        json.dump(dataset_prob, f, indent=4)


if __name__ == "__main__":
    from monai.utils import optional_import

    fire, _ = optional_import("fire")
    fire.Fire()
